
function [ tracks, discontinuedTracks, triangPredictions ] = loadInitialTracks( inputTracks, STB_params, cameraSystem)
% this function loads the tracks, generated by the 3D-particle tracking
% code and extracts the current particle positions and its estimated
% positions (kalmanFilter).
%
% OUTPUTS:
%   position:    contains x,y,z-columns
%   prediction:  reads the prediction-state of the kalmanFilter
%   kalFilter:   returns the kalmanFilter object
%   frame:       returns the frames, in which the tracks exist

minExist = 2;
% clear all inputTracks, that do not exist during the last minExist frames
endFrame  = traj_getFrameRange(convertToTraj(inputTracks));
notDeprecated = [];
for k = 1:size(inputTracks,2)
    % check frame existence-window
    try
        age = endFrame(2) - inputTracks(k).frames(end);
    catch
        notDeprecated(k) = false;
    end
    len = numel(inputTracks(k).frames);
    notDeprecated(k) = (age == 0) & (len >= minExist);
end

% only keep the notDeprecated-ones:
fprintf(1,'Keeping %d out of %d given tracks.\n',nnz(notDeprecated),size(inputTracks,2));
inputTracks = inputTracks(logical(notDeprecated));


tracks = struct('position',[],'frames',[], 'intensities', [], 'kalFilter', {}, 'converged', false, 'predictions', [], 'plotHandle', []);
% create a track-structure that will store discontinued tracks later.
discontinuedTracks = tracks; % will be used to store discontinued tracks
triangPredictions  = tracks; % will be used later to look for new tracks

imageArray = cell( size(inputTracks(1).position ,1) , 4 );
fprintf(1,'loadInitialTracks:: Loading Images...');

for frameNo = inputTracks(1).frames(1):inputTracks(1).frames(end)
    for camNo = 1:cameraSystem.nCameras
        imageArray{frameNo, camNo} = imread( sprintf(STB_params.imagePaths,camNo,frameNo) );
    end
end

fprintf(1,' [ done ]\n');
fprintf(1,'loadInitialTracks:: Processing Particle: ');
for k = 1:length(inputTracks)
    fprintf(1,'%04d/%04d',k, length(inputTracks));
    tracks(k).position  = inputTracks(k).position;
    tracks(k).frames    = inputTracks(k).frames;
    
    % initialize kalman filter with given options
    initPosition = tracks(k).position(1,:);        % initial position
    
    kalmanFilter = configureKalmanFilter(STB_params.kalmanFilter.Model, ...
        initPosition, STB_params.kalmanFilter.InitialEstimateError, STB_params.kalmanFilter.MotionNoise, STB_params.kalmanFilter.MeasurementNoise);
    tracks(k).kalFilter   = kalmanFilter;
    tracks(k).intensities(1,:) = STB_getInitialParticleIntensity( STB_params, initPosition, imageArray( tracks(k).frames(1), : ) , cameraSystem);
    tracks(k).predictions(1,:) = initPosition;
    % evolve the kalman filter with the (already known) positions and let
    % it "learn" how to predict further on
    for frame = tracks(k).frames(2):tracks(k).frames(end)
        prediction = predict(tracks(k).kalFilter);
        tracks(k).predictions(frame == tracks(k).frames,:) = prediction;
        newPosition = tracks(k).position(frame == tracks(k).frames,:);
        correct(tracks(k).kalFilter, newPosition);
        
        % get the particle intensity for that frame
        tracks(k).intensities(frame == tracks(k).frames,:) = STB_getInitialParticleIntensity( STB_params, newPosition, imageArray( frame , : ) , cameraSystem);
    end
    fprintf(1,'\b\b\b\b\b\b\b\b\b');
end
fprintf(1,' [ done ]\n');




end

